#!/usr/bin/env bash
set -euo pipefail

# --- Réglages ---
LOGFILE="/var/log/auto-upgrade.log"
LOCKFILE="/var/run/auto-upgrade.lock"

# --- Vérification root ---
if [ "$(id -u)" -ne 0 ]; then
  echo "Ce script doit être exécuté en root." >&2
  exit 1
fi

# --- Logging ---
umask 022
mkdir -p "$(dirname "$LOGFILE")"
if [ -t 1 ]; then
  exec > >(tee -a "$LOGFILE") 2>&1
else
  exec >>"$LOGFILE" 2>&1
fi

echo "===== $(date -Is) : démarrage auto-upgrade (AlmaLinux) ====="

# --- Lock pour éviter les collisions ---
exec 9>"$LOCKFILE"
if ! flock -n 9; then
  echo "Un autre auto-upgrade est déjà en cours, arrêt."
  exit 0
fi

# --- Fonction de détection reboot ---
reboot_required() {
  if command -v needs-restarting >/dev/null 2>&1; then
    if needs-restarting -r >/dev/null 2>&1; then
      return 1   # pas de reboot
    else
      return 0   # reboot requis
    fi
  fi

  local running latest
  running="$(uname -r)"
  latest="$(rpm -q kernel-core --qf '%{VERSION}-%{RELEASE}.%{ARCH}\n' 2>/dev/null | sort -V | tail -n1)"
  [ -z "$latest" ] && latest="$(rpm -q kernel --qf '%{VERSION}-%{RELEASE}.%{ARCH}\n' 2>/dev/null | sort -V | tail -n1)"
  if [ -n "$latest" ] && [ "$latest" != "$running" ]; then
    return 0
  fi
  return 1
}

# --- Étapes de mise à jour ---
echo "[1/4] dnf makecache --refresh"
dnf -y makecache --refresh

echo "[2/4] dnf upgrade --refresh"
dnf -y upgrade --refresh

echo "[3/4] dnf distro-sync"
dnf -y distro-sync

echo "[4/4] dnf autoremove"
dnf -y autoremove || true

# --- Reboot si nécessaire ---
if reboot_required; then
  echo "Reboot nécessaire (kernel ou libs critiques)."
  /sbin/reboot
else
  echo "Pas besoin de reboot."
fi

echo "===== $(date -Is) : fin auto-upgrade ====="
echo "--------------------------------------------------------"
exit 0